/*
 * Decompiled with CFR 0.152.
 */
package de.riwagis.riwadatatable.transaction;

import de.riwagis.riwadatatable.jdbc.JDBCConnectionInfo;
import de.riwagis.riwadatatable.transaction.Transaction;
import de.riwagis.riwadatatable.transaction.TransactionWatchDog;
import de.riwagis.util.exception.SystemException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Savepoint;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.dbutils.DbUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransactionManager {
    private static final Logger LOG = LoggerFactory.getLogger(TransactionManager.class);
    private final AtomicInteger transactionCounter = new AtomicInteger();
    private final Map<Integer, Transaction> transactionPool = new ConcurrentHashMap<Integer, Transaction>();
    private final Lock poolLock = new ReentrantLock();
    private final TransactionWatchDog watchDog;
    private boolean shutdown = false;

    public TransactionManager(long maxIdleTimeSeconds) {
        this.watchDog = new TransactionWatchDog(maxIdleTimeSeconds * 1000L, this);
        this.watchDog.start();
    }

    private void checkShutdown() {
        if (this.shutdown) {
            throw new IllegalStateException("TransactionManager has been shutdown.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Integer createTransaction(Connection conn, JDBCConnectionInfo connInfo) throws SystemException {
        this.checkShutdown();
        try {
            Integer transactionKey = this.transactionCounter.incrementAndGet();
            if (conn.getAutoCommit()) {
                conn.setAutoCommit(false);
            }
            Transaction transaction = new Transaction(transactionKey, conn, connInfo, this);
            this.poolLock.lock();
            try {
                this.transactionPool.put(transactionKey, transaction);
            }
            finally {
                this.poolLock.unlock();
            }
            return transactionKey;
        }
        catch (SQLException e) {
            throw new SystemException((Throwable)e);
        }
    }

    public JDBCConnectionInfo getConnectionInfo(Integer transactionKey) throws SystemException {
        this.checkShutdown();
        this.poolLock.lock();
        try {
            if (this.transactionPool.get(transactionKey) != null) {
                Transaction transaction = this.transactionPool.get(transactionKey);
                JDBCConnectionInfo jDBCConnectionInfo = transaction.getConnInfo();
                return jDBCConnectionInfo;
            }
            throw new SystemException(String.format("Transaction with key %s does not exist.", transactionKey));
        }
        finally {
            this.poolLock.unlock();
        }
    }

    private Connection checkOutConnectionInternal(Integer transactionKey) throws SystemException {
        if (this.transactionPool.get(transactionKey) != null) {
            Transaction transaction = this.transactionPool.get(transactionKey);
            if (transaction.isInUse()) {
                throw new SystemException(String.format("Transaction with key %s is already in use.", transactionKey));
            }
            transaction.setInUse(true);
            transaction.updateTimestamp();
            return transaction.getConnection();
        }
        throw new SystemException(String.format("Transaction with key %s does not exist.", transactionKey));
    }

    public Connection checkOutConnection(Integer transactionKey) throws SystemException {
        this.checkShutdown();
        this.poolLock.lock();
        try {
            Connection connection = this.checkOutConnectionInternal(transactionKey);
            return connection;
        }
        finally {
            this.poolLock.unlock();
        }
    }

    public boolean holdsTransaction(Integer transactionKey) {
        this.poolLock.lock();
        try {
            boolean bl = this.transactionPool.get(transactionKey) != null;
            return bl;
        }
        finally {
            this.poolLock.unlock();
        }
    }

    public void checkInConnection(Integer transactionKey) throws SystemException {
        this.checkInConnection(transactionKey, null);
    }

    public Savepoint checkInConnection(Integer transactionKey, String savePoint) throws SystemException {
        this.poolLock.lock();
        try {
            if (this.transactionPool.get(transactionKey) != null) {
                Transaction transaction = this.transactionPool.get(transactionKey);
                transaction.setInUse(false);
                transaction.updateTimestamp();
                Savepoint sp = savePoint != null && !savePoint.isEmpty() ? transaction.getConnection().setSavepoint(savePoint) : null;
                Savepoint savepoint = sp;
                return savepoint;
            }
            try {
                throw new SystemException(String.format("Transaction with key %s does not exist.", transactionKey));
            }
            catch (SQLException e) {
                throw new SystemException((Throwable)e);
            }
        }
        finally {
            this.poolLock.unlock();
        }
    }

    public void commitCloseTransaction(Integer transactionKey) throws SystemException {
        try {
            this.poolLock.lock();
            try {
                Connection conn = this.checkOutConnectionInternal(transactionKey);
                conn.commit();
                conn.close();
                this.transactionPool.remove(transactionKey);
            }
            finally {
                this.poolLock.unlock();
            }
        }
        catch (SystemException e) {
            throw e;
        }
        catch (SQLException e) {
            throw new SystemException("Error commiting transaction.", (Throwable)e);
        }
    }

    public void rollbackTransaction(Integer transactionKey, Savepoint sp) throws SystemException {
        try {
            Connection conn = this.checkOutConnection(transactionKey);
            conn.rollback(sp);
            this.checkInConnection(transactionKey);
        }
        catch (SystemException e) {
            throw e;
        }
        catch (SQLException e) {
            throw new SystemException("Error rolling back transaction to savepoint.", (Throwable)e);
        }
    }

    public void rollbackCloseTransaction(Integer transactionKey) throws SystemException {
        try {
            Connection conn = this.checkOutConnection(transactionKey);
            conn.rollback();
            conn.close();
            this.poolLock.lock();
            try {
                this.transactionPool.remove(transactionKey);
            }
            finally {
                this.poolLock.unlock();
            }
        }
        catch (SQLException e) {
            throw new SystemException("Error commiting transaction.", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void shutDownImmediately() {
        this.checkShutdown();
        this.shutdown = true;
        if (!this.transactionPool.isEmpty()) {
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        this.watchDog.shutdown();
        this.poolLock.lock();
        try {
            HashMap<Integer, Transaction> transactionPoolClone = new HashMap<Integer, Transaction>(this.transactionPool);
            for (Map.Entry<Integer, Transaction> entry : transactionPoolClone.entrySet()) {
                this.forceCloseAndRemoveTransactionInternal(entry);
            }
        }
        finally {
            this.poolLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeOldUnusedConnections(long maxIdleTime) {
        this.poolLock.lock();
        try {
            HashMap<Integer, Transaction> transactionPoolClone = new HashMap<Integer, Transaction>(this.transactionPool);
            for (Map.Entry<Integer, Transaction> entry : transactionPoolClone.entrySet()) {
                Transaction transaction = (Transaction)entry.getValue();
                if (transaction.isInUse()) continue;
                long transactionLastUsed = transaction.getTimestamp();
                if (System.currentTimeMillis() - transactionLastUsed <= maxIdleTime) continue;
                this.forceCloseAndRemoveTransactionInternal(entry);
                LOG.error(String.format("Should never reach here. Transaction on JDBC connection to %s , %s has been closed by watchdog.", transaction.getConnInfo().getUrl(), transaction.getConnInfo().getScheme()));
            }
        }
        finally {
            this.poolLock.unlock();
        }
    }

    private void forceCloseAndRemoveTransactionInternal(Map.Entry<Integer, Transaction> currTransaction) {
        Transaction transaction = currTransaction.getValue();
        Connection conn = transaction.getConnection();
        DbUtils.rollbackAndCloseQuietly((Connection)conn);
        this.transactionPool.remove(currTransaction.getKey());
    }
}

